<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

use App\Http\Services\DBClientConnection;

use App\Models\clientes\PlanVisitasSuh3100;
use App\Models\clientes\AuditoresVisitasSuh3100;
use App\Models\clientes\ServiciosVisitasSuh3100;
use App\Models\clientes\InstrumentosVisitasSuh3100;
use App\Models\clientes\AuxiliarIndicadoresVisitasSuh3100;
use App\Models\clientes\ParametrosVerifica;
use App\Models\hospitalia\Clientes;

use App\Models\base\User;
use App\Models\base\RepsPrestadores;
use App\Models\base\RepsPrestadoresSedes;

use App\Http\Controllers\API\MailController;
use App\Jobs\IndicadoresSuh3100Job;

use App\Http\Controllers\API\BoxController;

use Exception;

class PlanVisitasSuh3100Controller extends Controller
{
    private $cliente;

    function __construct() {
        // Crear la conexion temporal al esquema del cliente
        $this->middleware(function ($request, $next) {
            if (Auth::user() !== null) {
                $this->cliente = Auth::user()->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$this->cliente => $connection->getConnectionArray($this->cliente)]);
            }

            return $next($request);
        });
    }

    public function obtenerPlanVisitas(Request $request, $ano, $mes) {
        return PlanVisitasSuh3100::on($this->cliente)
                                ->selectRaw("
                                    plan_visitas_suh_3100.*, rpr.*, rps.*, esv,*,
                                    (
                                        select count(*) filter (where avs_visita_sincronizada = true)
                                        from auditores_visitas_suh_3100
                                        where plv_fk_id = plan_visitas_suh_3100.plv_pk_id
                                    ) as plv_sincronizada
                                ")
                                ->join('base.reps_prestadores as rpr', 'rpr_fk_id', 'rpr_pk_id')
                                ->join('base.reps_prestadores_sedes as rps', 'rps_fk_id', 'rps_pk_id')
                                ->leftJoin('base.estados_visita as esv', 'esv_fk_id', 'esv_pk_id')
                                ->where('plv_ano', $ano)
                                ->where('plv_mes', $mes)
                                ->where('plv_eliminada', false)
                                ->orderBy('plv_fecha_visita', 'asc')
                                ->orderBy('plv_pk_id', 'asc')
                                ->get()->toArray();
    }

    public function obtenerPlanVisitasAnoPrestador(Request $request, $ano, $prestador) {
        return PlanVisitasSuh3100::on($this->cliente)
                                ->join('base.reps_prestadores_sedes', 'rps_fk_id', 'rps_pk_id')
                                ->join('base.estados_visita', 'esv_fk_id', 'esv_pk_id')
                                ->where('plv_ano', $ano)
                                ->where('plan_visitas_suh_3100.rpr_fk_id', $prestador)
                                ->where('plv_eliminada', false)
                                ->where('plv_realizada', true)
                                ->orderBy('plv_fecha_visita', 'asc')
                                ->get()->toArray();
    }

    public function obtenerAuditoresVisita(Request $request, $idVisita) {
        $auditores = AuditoresVisitasSuh3100::on($this->cliente)
                                            ->selectRaw('auditores_visitas_suh_3100.*, usu_cargo')
                                            ->join('base.users', 'usu_fk_id', 'id')
                                            ->where('plv_fk_id', $idVisita)
                                            ->get()->toArray();

        // Obtener info usuarios hospitalia
        $idUsuarios = [];

        foreach ($auditores as $usu) {
            array_push($idUsuarios, $usu['usu_fk_id']);
        }

        $placeholders = implode(',', array_fill(0, count($idUsuarios), '?'));

        $infoUsuariosHospitalia = DB::connection('hospitalia')->select("
            select id, name, email, tdo_fk_id, tdo_descripcion_corta, usu_documento, usu_activo
            from users
            join tipos_documento on tdo_pk_id = tdo_fk_id
            where id in ($placeholders)
            order by 1
        ", $idUsuarios);

        // Completar la información
        foreach ($auditores as $key => $usu) {
            foreach ($infoUsuariosHospitalia as $usuHosp) {
                if (intval($usuHosp->id) === intval($usu['usu_fk_id'])) {
                    $auditores[$key]['name'] = $usuHosp->name;
                    $auditores[$key]['email'] = $usuHosp->email;
                    $auditores[$key]['tdo_fk_id'] = $usuHosp->tdo_fk_id;
                    $auditores[$key]['tdo_descripcion_corta'] = $usuHosp->tdo_descripcion_corta;
                    $auditores[$key]['usu_documento'] = $usuHosp->usu_documento;
                    break;
                }
            }
        }

        return $auditores;
    }

    public function obtenerServiciosVisita(Request $request, $idVisita) {
        return ServiciosVisitasSuh3100::on($this->cliente)
                                     ->selectRaw('servicios_visitas_suh_3100.*, grs_descripcion')
                                     ->join('base.grupos_servicio', 'grs_pk_id', 'grs_fk_id')
                                     ->where('plv_fk_id', $idVisita)
                                     ->orderBy('grs_descripcion')
                                     ->orderBy('svs_nombre_servicio')
                                     ->get()->toArray();
    }

    public function obtenerSoportesVisita(Request $request, $idVisita) {
        // Obtener los servicios habilitados para la visita
        $servicios = ServiciosVisitasSuh3100::on($this->cliente)
                                            ->selectRaw('svs_pk_id, grs_descripcion, svs_codigo_servicio, svs_nombre_servicio')
                                            ->join('base.grupos_servicio', 'grs_fk_id', 'grs_pk_id')
                                            ->where('plv_fk_id', $idVisita)
                                            ->orderBy('svs_pk_id')
                                            ->get()->toArray();

        // Obtener los archivos almacenados en el Box
        $idCarpetaVisita = PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $idVisita)->pluck('plv_box_id_visita')[0];
        $soportes = [];

        $boxController = new BoxController();
        $boxController->generarToken();
        $responseFolder = json_decode(json_encode($boxController->listarArchivosFolder($idCarpetaVisita)), true);

        if ($responseFolder['estado']) {
            $archivosBox = $responseFolder['data'];

            foreach ($archivosBox['entries'] as $sop) {
                if ($sop['name'] !== 'inf.pdf') {
                    $soporte = [];

                    $soporte['id'] = $sop['id'];
                    $soporte['nombre'] = explode('.', $sop['name'])[0];

                    if ($soporte['nombre'] !== 'spa') {
                        $infoServicio = array_filter($servicios, function($srv) use($sop) {
                            return intval($srv['svs_pk_id']) == intval($sop['name']);
                        });

                        $infoServicio = array_values($infoServicio);

                        if (count($infoServicio) > 0) {
                            $soporte['grupo_servicio'] = $infoServicio[0]['grs_descripcion'];
                            $soporte['codigo_servicio'] = $infoServicio[0]['svs_codigo_servicio'];
                            $soporte['nombre_servicio'] = $infoServicio[0]['svs_nombre_servicio'];
                        }
                    } else {
                        $soporte['grupo_servicio'] = '';
                        $soporte['codigo_servicio'] = '000AAA';
                        $soporte['nombre_servicio'] = 'SUFICIENCIA PATRIMONIAL Y FINANCIERA';
                    }

                    array_push($soportes, $soporte);
                }
            }
        } else {
            throw new Exception('No se pudo listar los archivos de la carpeta '.$idCarpetaVisita);
        }

        return $soportes;
    }

    public function obtenerSoportesEspecificoVisita(Request $request, $idSoporte) {
        $boxController = new BoxController();
        $boxController->generarToken();

        $descarga = $boxController->descargarArchivo($idSoporte);

        if ($descarga['estado']) {
            return $descarga['file'];
        } else {
            throw new Exception('No se pudo descargar el archivo '.$idSoporte);
        }
    }

    public function descargarInformeFinalVisita(Request $request, $idArchivo) {
        $boxController = new BoxController();
        $boxController->generarToken();

        $descarga = $boxController->descargarArchivo($idArchivo);

        if ($descarga['estado']) {
            return $descarga['file'];
        } else {
            throw new Exception('No se pudo descargar el archivo '.$idArchivo);
        }
    }

    public function buscarAsignacionPrestador(Request $request, $ano, $mes, $idPrestador, $idSede) {
        return PlanVisitasSuh3100::on($this->cliente)
                                ->where('plv_ano', $ano)
                                ->where('plv_mes', $mes)
                                ->where('plv_eliminada', false)
                                ->where('rpr_fk_id', $idPrestador)
                                ->where('rps_fk_id', $idSede)
                                ->count();
    }

    public function crearPlanVisitas(Request $request) {
        $id = null;
        $codigoResponsable = strtoupper(Str::random(4));

        $ano = $request->input('ano');
        $mes = $request->input('mes');

        DB::connection($this->cliente)->transaction(function() use($request, &$id, $ano, $mes, $codigoResponsable) {
            $fecha = $request->input('fechaVisita');
            $idPrestador = $request->input('idPrestador');
            $idSede = $request->input('idSede');
            $auditoresVisita = $request->input('auditoresVisita');
            $servicios = $request->input('servicios');
            $comentario = $request->input('comentario');

            $fechaVisita = null;

            if ($fecha === null) {
                $fechaVisita = date("Y-m-d");
                $fechaVisita = date("Y-m-d", strtotime($fechaVisita . " +1 day"));
            } else {
                $fechaVisita = $fecha;
            }

            $id = PlanVisitasSuh3100::on($this->cliente)->create([
                'plv_ano' => $ano,
                'plv_mes' => $mes,
                'plv_fecha_visita' => $fechaVisita,
                'rpr_fk_id' => $idPrestador,
                'rps_fk_id' => $idSede,
                'plv_comentario' => $comentario,
                'plv_codigo_visita' => $codigoResponsable,
                'plv_creada_por' => Auth::user()->id
            ])->plv_pk_id;

            $idCoordinador = null;

            foreach ($auditoresVisita as $avr) {
                AuditoresVisitasSuh3100::on($this->cliente)->create([
                    'plv_fk_id' => $id,
                    'avs_ano' => $ano,
                    'avs_mes' => $mes,
                    'usu_fk_id' => $avr['usu_fk_id'],
                    'avs_responsable' => $avr['avs_responsable']
                ]);

                if ($avr['avs_responsable']) {
                    $idCoordinador = $avr['usu_fk_id'];
                }
            }

            foreach ($servicios as $srv) {
                $srv['plv_fk_id'] = $id;
                ServiciosVisitasSuh3100::on($this->cliente)->create($srv);
            }

            // Crear la carpeta donde se guardarán los soportes de la visita
            $idCarpetaSuh3100 = ParametrosVerifica::on($this->cliente)->first()->pav_box_id_suh_3100;
            $boxController = new BoxController();

            $token = $boxController->generarToken();

            if ($token['estado']) {
                $estadoCarpeta = $boxController->crearFolder($id, $idCarpetaSuh3100);

                if ($estadoCarpeta['estado']) {
                    PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $id)->update([
                        'plv_box_id_visita' => $estadoCarpeta['id']
                    ]);
                } else {
                    throw new Exception('No fue posible crear la carpeta de la visita SUH 3100 con id '.$id);
                }
            } else {
                throw new Exception('No fue posible obtener un token para interactuar con BOX');
            }

            // Envío de correo al coordinador
            /*$correoCoordinador = User::where('id', $idCoordinador)->pluck('email')[0];

            if ($correoCoordinador !== null) {
                $nombrePrestador = RepsPrestadores::where('rpr_pk_id', $idPrestador)->pluck('rpr_razon_social')[0];
                $nombreSede = RepsPrestadoresSedes::where('rps_pk_id', $idSede)->pluck('rps_nombre_sede')[0];

                (new MailController)->enviarNotificacionVisitaRias($correoCoordinador, $nombrePrestador, $nombreSede, $codigoResponsable);
            }*/
        });

        IndicadoresSuh3100Job::dispatch($ano, $mes)->onQueue('indicadores-suh-3100');

        return [$id, $codigoResponsable];
    }

    public function actualizarPlanVisitas(Request $request, $idVisita) {
        DB::connection($this->cliente)->transaction(function() use($request, &$idVisita) {
            $auditoresVisita = $request->input('auditoresVisita');
            $servicios = $request->input('servicios');
            $comentario = $request->input('comentario');

            PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $idVisita)->update(['plv_comentario' => $comentario]);

            foreach ($auditoresVisita as $avr) {
                if (isset($avr['avs_borrado']) && $avr['avs_borrado']) {
                    AuditoresVisitasSuh3100::on($this->cliente)->where('avs_pk_id', $avr['avs_pk_id'])->delete();
                }
            }

            foreach ($servicios as $srv) {
                $srv['plv_fk_id'] = $idVisita;

                if ($srv['svs_pk_id'] === null) {
                    ServiciosVisitasSuh3100::on($this->cliente)->create($srv);
                } else {
                    unset($srv['grs_descripcion']);
                    ServiciosVisitasSuh3100::on($this->cliente)->where('svs_pk_id', $srv['svs_pk_id'])->update($srv);
                }
            }
        });
    }

    public function eliminarPrestadorPlanVisitas(Request $request, $idVisita) {
        $ano = $request->input('ano');
        $mes = $request->input('mes');

        PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $idVisita)
        ->update([
            'plv_eliminada' => true,
            'plv_borrada_por' => Auth::user()->id
        ]);

        IndicadoresSuh3100Job::dispatch($ano, $mes)->onQueue('indicadores-suh-3100');
    }

    public function actualizarSincronizacionVisita(Request $request, $idVisita) {
        $estado = $request->input('estado');

        AuditoresVisitasSuh3100::on($this->cliente)
                            ->where('plv_fk_id', $idVisita)
                            ->where('usu_fk_id', Auth::user()->id)
                            ->update(['avs_visita_sincronizada' => $estado]);
    }

    public function finalizarVisita(Request $request) {
        $usuario = Auth::user()->id;
        $infoVisita = json_decode($request->infoVisita, true);
        $instrumentos = json_decode($request->instrumentos, true);

        $visitasSinFinalizar = 0;

        DB::connection($this->cliente)->transaction(function () use ($request, $usuario, $infoVisita, $instrumentos, &$visitasSinFinalizar) {
            // Guardar instrumentos
            InstrumentosVisitasSuh3100::on($this->cliente)->where('plv_fk_id', $infoVisita['plv_id_db'])->where('usu_fk_id', $usuario)->delete();

            $this->guardarInstrumentosVisita($infoVisita, $usuario, $instrumentos);

            // Revisar si se puede dar finalizar la visita
            $visitasSinFinalizar = AuditoresVisitasSuh3100::on($this->cliente)
                                                        ->where('plv_fk_id', $infoVisita['plv_id_db'])
                                                        ->where('usu_fk_id', '!=', $usuario)
                                                        ->where('avs_finalizada', false)
                                                        ->count();

            // Preparar la nueva información de la visita
            $nuevaInfoVisita = [
                'plv_realizada' => intval($visitasSinFinalizar) === 0 ? true : false,
                'plv_fecha_cierre_visita' => intval($visitasSinFinalizar) === 0 ? 'now()' : null
            ];

            if (intval($visitasSinFinalizar) === 0) {
                $nuevaInfoVisita['plv_guardado_nube'] = true;
            }

            if ($infoVisita['plv_responsable']) {
                $nuevaInfoVisita['esv_fk_id'] = $infoVisita['esv_fk_id'];
                $nuevaInfoVisita['plv_comentario_cierre'] = $infoVisita['plv_comentario_cierre'];
            }

            $instrumentoSufPatrimonial = isset($instrumentos['spa']['suficiencia-patrimonial']) ? json_encode($instrumentos['spa']['suficiencia-patrimonial']) : null;

            if ($instrumentoSufPatrimonial !== null) {
                $nuevaInfoVisita['plv_ins_suf_patrimonial'] = $instrumentoSufPatrimonial;
            }

            // Procesar los indicadores
            $this->consolidarAuxiliarIndicadores($usuario, $infoVisita);

            // Actualizar la asignación del verificador
            AuditoresVisitasSuh3100::on($this->cliente)
                                    ->where('plv_fk_id', $infoVisita['plv_id_db'])
                                    ->where('usu_fk_id', $usuario)
                                    ->update(['avs_finalizada' => true]);

            // Actualizar la visita
            PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $infoVisita['plv_id_db'])->update($nuevaInfoVisita);

            // Carga de soportes en el Box
            $almacenamientoNube = Clientes::on('hospitalia')->where('cli_pk_id', $this->cliente)->pluck('cli_almacenamiento_nube')[0];

            if ($almacenamientoNube) {
                $soportes = $request->file('soportes');

                if ($soportes !== null) {
                    $idCarpeta = PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $infoVisita['plv_id_db'])->pluck('plv_box_id_visita')[0];

                    $boxController = new BoxController();
                    $boxController->generarToken();

                    foreach ($soportes as $file) {
                        $nombreSoporte = $file->getClientOriginalName().'.pdf';
                        $estado = $boxController->cargarArchivo($idCarpeta, $file, $nombreSoporte);

                        if (!$estado['estado']) {
                            throw new Exception('No fue posible cargar el archivo correspondiente al servicio '.$nombreSoporte);
                        }
                    }
                }
            }
        });

        // Calcular los indicadores mediante el job si se puede finalizar la visita
        if (intval($visitasSinFinalizar) === 0) {
            $this->consolidarIndicadoresGeneralesVisita($infoVisita['plv_id_db']);
            IndicadoresSuh3100Job::dispatch($infoVisita['plv_ano'], $infoVisita['plv_mes'])->onQueue('indicadores-suh-3100');
        }
    }

    private function guardarInstrumentosVisita(&$infoVisita, $usuario, $instrumentos) {
        foreach ($instrumentos as $key => $value) {
            if ($key !== 'spa') {
                InstrumentosVisitasSuh3100::on($this->cliente)->create([
                    'plv_fk_id' => $infoVisita['plv_id_db'],
                    'ivs_ano' => $infoVisita['plv_ano'],
                    'ivs_mes' => $infoVisita['plv_mes'],
                    'usu_fk_id' => $usuario,
                    'svs_fk_id' => $key,

                    'ivs_talento_humano' => isset($value['talento-humano']) ? json_encode($value['talento-humano']) : null,
                    'ivs_infraestructura' => isset($value['infraestructura']) ? json_encode($value['infraestructura']) : null,
                    'ivs_dotacion' => isset($value['dotacion']) ? json_encode($value['dotacion']) : null,
                    'ivs_medicamentos' => isset($value['med-disp-ins']) ? json_encode($value['med-disp-ins']) : null,
                    'ivs_procesos_prioritarios' => isset($value['proc-prioritarios']) ? json_encode($value['proc-prioritarios']) : null,
                    'ivs_historia_clinica' => isset($value['hist-clinica']) ? json_encode($value['hist-clinica']) : null,
                    'ivs_interdependencia' => isset($value['interdependencia']) ? json_encode($value['interdependencia']) : null,

                    'ivs_cons_ext_general' => isset($value['cons-ext-general']) ? json_encode($value['cons-ext-general']) : null,
                    'ivs_cons_ext_especializada' => isset($value['cons-ext-especializada']) ? json_encode($value['cons-ext-especializada']) : null,
                    'ivs_cons_ext_vacunacion' => isset($value['ce-vacunacion']) ? json_encode($value['ce-vacunacion']) : null,
                    'ivs_cons_ext_sst' => isset($value['ce-sst']) ? json_encode($value['ce-sst']) : null,

                    'ivs_ad_terapias' => isset($value['ad-terapias']) ? json_encode($value['ad-terapias']) : null,
                    'ivs_ad_farmaceutico' => isset($value['ad-farmaceutico']) ? json_encode($value['ad-farmaceutico']) : null,
                    'ivs_ad_rad_odontologica' => isset($value['ad-rad-odontologica']) ? json_encode($value['ad-rad-odontologica']) : null,
                    'ivs_ad_img_diag_ionizantes' => isset($value['ad-img-diagnos-ion']) ? json_encode($value['ad-img-diagnos-ion']) : null,
                    'ivs_ad_img_diag_no_ionizantes' => isset($value['ad-img-diagnos-no-ion']) ? json_encode($value['ad-img-diagnos-no-ion']) : null,
                    'ivs_ad_medicina_nuclear' => isset($value['ad-med-nuclear']) ? json_encode($value['ad-med-nuclear']) : null,
                    'ivs_ad_radioterapia' => isset($value['ad-radioterapia']) ? json_encode($value['ad-radioterapia']) : null,
                    'ivs_ad_quimioterapia' => isset($value['ad-quimioterapia']) ? json_encode($value['ad-quimioterapia']) : null,
                    'ivs_ad_diag_vascular' => isset($value['ad-diag-vascular']) ? json_encode($value['ad-diag-vascular']) : null,
                    'ivs_ad_hemodinamia' => isset($value['ad-hemodinamia']) ? json_encode($value['ad-hemodinamia']) : null,
                    'ivs_ad_gest_pretansfusional' => isset($value['ad-pre-transfusional']) ? json_encode($value['ad-pre-transfusional']) : null,
                    'ivs_ad_muestras_lab_clinico' => isset($value['ad-muestras-lab-clinico']) ? json_encode($value['ad-muestras-lab-clinico']) : null,
                    'ivs_ad_laboratorio_clinico' => isset($value['ad-lab-clinico']) ? json_encode($value['ad-lab-clinico']) : null,
                    'ivs_ad_muestras_cuello_uterino' => isset($value['ad-muestras-gineco']) ? json_encode($value['ad-muestras-gineco']) : null,
                    'ivs_ad_lab_citologias' => isset($value['ad-lab-citologias']) ? json_encode($value['ad-lab-citologias']) : null,
                    'ivs_ad_lab_histotecnologia' => isset($value['ad-lab-histotecnologia']) ? json_encode($value['ad-lab-histotecnologia']) : null,
                    'ivs_ad_patologia' => isset($value['ad-patologia']) ? json_encode($value['ad-patologia']) : null,
                    'ivs_ad_dialisis' => isset($value['ad-dialisis']) ? json_encode($value['ad-dialisis']) : null,

                    'ivs_int_hospitalizacion' => isset($value['int-hospitalizacion']) ? json_encode($value['int-hospitalizacion']) : null,
                    'ivs_int_hospitalizacion_pc' => isset($value['int-hospitalizacion-pc']) ? json_encode($value['int-hospitalizacion-pc']) : null,
                    'ivs_int_cuid_bas_neonatal' => isset($value['int-ciud-basico-neonatal']) ? json_encode($value['int-ciud-basico-neonatal']) : null,
                    'ivs_int_cuid_inm_neonatal' => isset($value['int-ciud-inter-neonatal']) ? json_encode($value['int-ciud-inter-neonatal']) : null,
                    'ivs_int_cuid_int_neonatal' => isset($value['int-ciud-intens-neonatal']) ? json_encode($value['int-ciud-intens-neonatal']) : null,
                    'ivs_int_cuid_inm_pediatrico' => isset($value['int-ciud-inter-pediat']) ? json_encode($value['int-ciud-inter-pediat']) : null,
                    'ivs_int_cuid_int_pediatrico' => isset($value['int-ciud-intens-pediat']) ? json_encode($value['int-ciud-intens-pediat']) : null,
                    'ivs_int_cuid_inm_adultos' => isset($value['int-ciud-inter-adultos']) ? json_encode($value['int-ciud-inter-adultos']) : null,
                    'ivs_int_cuid_int_adultos' => isset($value['int-ciud-intens-adultos']) ? json_encode($value['int-ciud-intens-adultos']) : null,
                    'ivs_int_hospitalizacion_sm' => isset($value['int-hosp-salud-mental']) ? json_encode($value['int-hosp-salud-mental']) : null,
                    'ivs_int_hospitalizacion_parcial' => isset($value['int-hosp-parcial']) ? json_encode($value['int-hosp-parcial']) : null,
                    'ivs_int_cuid_bas_spa' => isset($value['int-cb-spa']) ? json_encode($value['int-cb-spa']) : null,

                    'ivs_quir_cirugia' => isset($value['cirugia']) ? json_encode($value['cirugia']) : null,

                    'ivs_ai_urgencias' => isset($value['ai-urgencias']) ? json_encode($value['ai-urgencias']) : null,
                    'ivs_ai_transp_asistencial' => isset($value['ai-trans-asistencial']) ? json_encode($value['ai-trans-asistencial']) : null,
                    'ivs_ai_atn_prehospitalaria' => isset($value['ai-atn-prehospitalaria']) ? json_encode($value['ai-atn-prehospitalaria']) : null,
                    'ivs_ai_atencion_parto' => isset($value['ai-atn-parto']) ? json_encode($value['ai-atn-parto']) : null
                ]);
            }
        }
    }

    private function consolidarAuxiliarIndicadores($usuario, $infoVisita) {
        // Preparativos previos
        $estandares = ['todos', 'thu', 'inf', 'dot', 'med', 'ppr', 'hcl', 'ind'];
        $campos = ['total_preguntas', 'total_cumple', 'total_cumple_na', 'total_na', 'total_no_cumple', 'total_no_verificado'];

        $plantillaIndicadores = [
            'plv_fk_id' => $infoVisita['plv_id_db'],
            'ais_ano' => $infoVisita['plv_ano'],
            'ais_mes' => $infoVisita['plv_mes'],
            'usu_fk_id' => $usuario,
            'svs_fk_id' => null
        ];

        foreach ($estandares as $est) {
            if ($est === 'todos') {
                foreach ($campos as $cmp) {
                    $plantillaIndicadores['ais_'.$cmp] = 0;
                }
            } else {
                foreach ($campos as $cmp) {
                    $plantillaIndicadores['ais_'.$est.'_'.$cmp] = 0;
                }
            }
        }

        AuxiliarIndicadoresVisitasSuh3100::on($this->cliente)
                                         ->where('plv_fk_id', $infoVisita['plv_id_db'])
                                         ->where('usu_fk_id', $usuario)
                                         ->delete();

        // Extracción de los indicadores generales
        if (isset($infoVisita['indicadores']) && isset($infoVisita['indicadores']['cumplimiento_general']) &&
            intval($infoVisita['indicadores']['cumplimiento_general']['plv_total_preguntas']) > 0) {
            $indicadores = $plantillaIndicadores;

            foreach ($estandares as $est) {
                if ($est === 'todos') {
                    foreach ($campos as $cmp) {
                        $indicadores['ais_'.$cmp] = $infoVisita['indicadores']['cumplimiento_general']['plv_'.$cmp];
                    }
                } else {
                    foreach ($campos as $cmp) {
                        $indicadores['ais_'.$est.'_'.$cmp] = $infoVisita['indicadores']['cumplimiento_general']['plv_'.$est.'_'.$cmp];
                    }
                }
            }

            AuxiliarIndicadoresVisitasSuh3100::on($this->cliente)->create($indicadores);

            // Extracción de los indicadores por servicio
            foreach ($infoVisita['indicadores']['cumplimiento_servicio'] as $key => $value) {
                $indicadores = $plantillaIndicadores;
                $indicadores['svs_fk_id'] = $key;

                foreach ($estandares as $est) {
                    if ($est === 'todos') {
                        foreach ($campos as $cmp) {
                            $indicadores['ais_'.$cmp] = $value['plv_'.$cmp];
                        }
                    } else {
                        foreach ($campos as $cmp) {
                            $indicadores['ais_'.$est.'_'.$cmp] = $value['plv_'.$est.'_'.$cmp];
                        }
                    }
                }

                AuxiliarIndicadoresVisitasSuh3100::on($this->cliente)->create($indicadores);
            }
        }
    }

    private function consolidarIndicadoresGeneralesVisita($idVisita) {
        $consolidado = AuxiliarIndicadoresVisitasSuh3100::on($this->cliente)
        ->selectRaw("
            sum(ais_total_preguntas) as plv_total_preguntas,
            sum(ais_total_cumple) as plv_total_cumple,
            sum(ais_total_cumple_na) as plv_total_cumple_na,
            sum(ais_total_na) as plv_total_na,
            sum(ais_total_no_cumple) as plv_total_no_cumple,
            sum(ais_total_no_verificado) as plv_total_no_verificado,

            sum(ais_thu_total_preguntas) as plv_thu_total_preguntas,
            sum(ais_thu_total_cumple) as plv_thu_total_cumple,
            sum(ais_thu_total_cumple_na) as plv_thu_total_cumple_na,
            sum(ais_thu_total_na) as plv_thu_total_na,
            sum(ais_thu_total_no_cumple) as plv_thu_total_no_cumple,
            sum(ais_thu_total_no_verificado) as plv_thu_total_no_verificado,

            sum(ais_inf_total_preguntas) as plv_inf_total_preguntas,
            sum(ais_inf_total_cumple) as plv_inf_total_cumple,
            sum(ais_inf_total_cumple_na) as plv_inf_total_cumple_na,
            sum(ais_inf_total_na) as plv_inf_total_na,
            sum(ais_inf_total_no_cumple) as plv_inf_total_no_cumple,
            sum(ais_inf_total_no_verificado) as plv_inf_total_no_verificado,

            sum(ais_inf_total_preguntas) as plv_inf_total_preguntas,
            sum(ais_inf_total_cumple) as plv_inf_total_cumple,
            sum(ais_inf_total_cumple_na) as plv_inf_total_cumple_na,
            sum(ais_inf_total_na) as plv_inf_total_na,
            sum(ais_inf_total_no_cumple) as plv_inf_total_no_cumple,
            sum(ais_inf_total_no_verificado) as plv_inf_total_no_verificado,

            sum(ais_dot_total_preguntas) as plv_dot_total_preguntas,
            sum(ais_dot_total_cumple) as plv_dot_total_cumple,
            sum(ais_dot_total_cumple_na) as plv_dot_total_cumple_na,
            sum(ais_dot_total_na) as plv_dot_total_na,
            sum(ais_dot_total_no_cumple) as plv_dot_total_no_cumple,
            sum(ais_dot_total_no_verificado) as plv_dot_total_no_verificado,

            sum(ais_med_total_preguntas) as plv_med_total_preguntas,
            sum(ais_med_total_cumple) as plv_med_total_cumple,
            sum(ais_med_total_cumple_na) as plv_med_total_cumple_na,
            sum(ais_med_total_na) as plv_med_total_na,
            sum(ais_med_total_no_cumple) as plv_med_total_no_cumple,
            sum(ais_med_total_no_verificado) as plv_med_total_no_verificado,

            sum(ais_ppr_total_preguntas) as plv_ppr_total_preguntas,
            sum(ais_ppr_total_cumple) as plv_ppr_total_cumple,
            sum(ais_ppr_total_cumple_na) as plv_ppr_total_cumple_na,
            sum(ais_ppr_total_na) as plv_ppr_total_na,
            sum(ais_ppr_total_no_cumple) as plv_ppr_total_no_cumple,
            sum(ais_ppr_total_no_verificado) as plv_ppr_total_no_verificado,

            sum(ais_hcl_total_preguntas) as plv_hcl_total_preguntas,
            sum(ais_hcl_total_cumple) as plv_hcl_total_cumple,
            sum(ais_hcl_total_cumple_na) as plv_hcl_total_cumple_na,
            sum(ais_hcl_total_na) as plv_hcl_total_na,
            sum(ais_hcl_total_no_cumple) as plv_hcl_total_no_cumple,
            sum(ais_hcl_total_no_verificado) as plv_hcl_total_no_verificado,

            sum(ais_ind_total_preguntas) as plv_ind_total_preguntas,
            sum(ais_ind_total_cumple) as plv_ind_total_cumple,
            sum(ais_ind_total_cumple_na) as plv_ind_total_cumple_na,
            sum(ais_ind_total_na) as plv_ind_total_na,
            sum(ais_ind_total_no_cumple) as plv_ind_total_no_cumple,
            sum(ais_ind_total_no_verificado) as plv_ind_total_no_verificado
        ")
        ->where('plv_fk_id', $idVisita)->whereNull('svs_fk_id')
        ->get()->toArray();

        if (count($consolidado) > 0) {
            PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $idVisita)->update([
                'plv_total_preguntas' => $consolidado[0]['plv_total_preguntas'],
                'plv_total_cumple' => $consolidado[0]['plv_total_cumple'],
                'plv_total_cumple_na' => $consolidado[0]['plv_total_cumple_na'],
                'plv_total_na' => $consolidado[0]['plv_total_na'],
                'plv_total_no_cumple' => $consolidado[0]['plv_total_no_cumple'],
                'plv_total_no_verificado' => $consolidado[0]['plv_total_no_verificado'],

                'plv_thu_total_preguntas' => $consolidado[0]['plv_thu_total_preguntas'],
                'plv_thu_total_cumple' => $consolidado[0]['plv_thu_total_cumple'],
                'plv_thu_total_cumple_na' => $consolidado[0]['plv_thu_total_cumple_na'],
                'plv_thu_total_na' => $consolidado[0]['plv_thu_total_na'],
                'plv_thu_total_no_cumple' => $consolidado[0]['plv_thu_total_no_cumple'],
                'plv_thu_total_no_verificado' => $consolidado[0]['plv_thu_total_no_verificado'],

                'plv_inf_total_preguntas' => $consolidado[0]['plv_inf_total_preguntas'],
                'plv_inf_total_cumple' => $consolidado[0]['plv_inf_total_cumple'],
                'plv_inf_total_cumple_na' => $consolidado[0]['plv_inf_total_cumple_na'],
                'plv_inf_total_na' => $consolidado[0]['plv_inf_total_na'],
                'plv_inf_total_no_cumple' => $consolidado[0]['plv_inf_total_no_cumple'],
                'plv_inf_total_no_verificado' => $consolidado[0]['plv_inf_total_no_verificado'],

                'plv_dot_total_preguntas' => $consolidado[0]['plv_dot_total_preguntas'],
                'plv_dot_total_cumple' => $consolidado[0]['plv_dot_total_cumple'],
                'plv_dot_total_cumple_na' => $consolidado[0]['plv_dot_total_cumple_na'],
                'plv_dot_total_na' => $consolidado[0]['plv_dot_total_na'],
                'plv_dot_total_no_cumple' => $consolidado[0]['plv_dot_total_no_cumple'],
                'plv_dot_total_no_verificado' => $consolidado[0]['plv_dot_total_no_verificado'],

                'plv_med_total_preguntas' => $consolidado[0]['plv_med_total_preguntas'],
                'plv_med_total_cumple' => $consolidado[0]['plv_med_total_cumple'],
                'plv_med_total_cumple_na' => $consolidado[0]['plv_med_total_cumple_na'],
                'plv_med_total_na' => $consolidado[0]['plv_med_total_na'],
                'plv_med_total_no_cumple' => $consolidado[0]['plv_med_total_no_cumple'],
                'plv_med_total_no_verificado' => $consolidado[0]['plv_med_total_no_verificado'],

                'plv_ppr_total_preguntas' => $consolidado[0]['plv_ppr_total_preguntas'],
                'plv_ppr_total_cumple' => $consolidado[0]['plv_ppr_total_cumple'],
                'plv_ppr_total_cumple_na' => $consolidado[0]['plv_ppr_total_cumple_na'],
                'plv_ppr_total_na' => $consolidado[0]['plv_ppr_total_na'],
                'plv_ppr_total_no_cumple' => $consolidado[0]['plv_ppr_total_no_cumple'],
                'plv_ppr_total_no_verificado' => $consolidado[0]['plv_ppr_total_no_verificado'],

                'plv_hcl_total_preguntas' => $consolidado[0]['plv_hcl_total_preguntas'],
                'plv_hcl_total_cumple' => $consolidado[0]['plv_hcl_total_cumple'],
                'plv_hcl_total_cumple_na' => $consolidado[0]['plv_hcl_total_cumple_na'],
                'plv_hcl_total_na' => $consolidado[0]['plv_hcl_total_na'],
                'plv_hcl_total_no_cumple' => $consolidado[0]['plv_hcl_total_no_cumple'],
                'plv_hcl_total_no_verificado' => $consolidado[0]['plv_hcl_total_no_verificado'],

                'plv_ind_total_preguntas' => $consolidado[0]['plv_ind_total_preguntas'],
                'plv_ind_total_cumple' => $consolidado[0]['plv_ind_total_cumple'],
                'plv_ind_total_cumple_na' => $consolidado[0]['plv_ind_total_cumple_na'],
                'plv_ind_total_na' => $consolidado[0]['plv_ind_total_na'],
                'plv_ind_total_no_cumple' => $consolidado[0]['plv_ind_total_no_cumple'],
                'plv_ind_total_no_verificado' => $consolidado[0]['plv_ind_total_no_verificado']
            ]);
        }
    }

    public function obtenerResumenServicioEstandar(Request $request, $idVisita) {
        return DB::select('
            select grs_fk_id,
                   grs_descripcion,
                   svs_codigo_servicio,
                   svs_nombre_servicio,

                   ais_total_preguntas,
                   ais_total_cumple,
                   ais_total_cumple_na,
                   ais_total_na,
                   ais_total_no_cumple,
                   ais_total_no_verificado,

                   ais_thu_total_preguntas,
                   ais_thu_total_cumple,
                   ais_thu_total_cumple_na,
                   ais_thu_total_na,
                   ais_thu_total_no_cumple,
                   ais_thu_total_no_verificado,

                   ais_inf_total_preguntas,
                   ais_inf_total_cumple,
                   ais_inf_total_cumple_na,
                   ais_inf_total_na,
                   ais_inf_total_no_cumple,
                   ais_inf_total_no_verificado,

                   ais_dot_total_preguntas,
                   ais_dot_total_cumple,
                   ais_dot_total_cumple_na,
                   ais_dot_total_na,
                   ais_dot_total_no_cumple,
                   ais_dot_total_no_verificado,

                   ais_med_total_preguntas,
                   ais_med_total_cumple,
                   ais_med_total_cumple_na,
                   ais_med_total_na,
                   ais_med_total_no_cumple,
                   ais_med_total_no_verificado,

                   ais_ppr_total_preguntas,
                   ais_ppr_total_cumple,
                   ais_ppr_total_cumple_na,
                   ais_ppr_total_na,
                   ais_ppr_total_no_cumple,
                   ais_ppr_total_no_verificado,

                   ais_hcl_total_preguntas,
                   ais_hcl_total_cumple,
                   ais_hcl_total_cumple_na,
                   ais_hcl_total_na,
                   ais_hcl_total_no_cumple,
                   ais_hcl_total_no_verificado,

                   ais_ind_total_preguntas,
                   ais_ind_total_cumple,
                   ais_ind_total_cumple_na,
                   ais_ind_total_na,
                   ais_ind_total_no_cumple,
                   ais_ind_total_no_verificado
            from "'.$this->cliente.'".auxiliar_indicadores_visitas_suh_3100 as ais
            left join "'.$this->cliente.'".servicios_visitas_suh_3100 as svs on svs_fk_id = svs_pk_id
            left join "base".grupos_servicio as grs on grs_fk_id = grs_pk_id
            where ais.plv_fk_id = ?
            order by 2 nulls first, 4 nulls first
        ', [$idVisita]);
    }

    public function cargarInformeFinalVisita(Request $request) {
        $idArchivo = null;

        $idVisita = $request->idVisita;
        $idInforme = $request->idArchivo;
        $archivo = $request->file('informe');

        $idCarpeta = PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $idVisita)->pluck('plv_box_id_visita')[0];

        $boxController = new BoxController();
        $boxController->generarToken();

        if (trim(strval($idInforme)) === '') {
            $nombreSoporte = 'inf.pdf';
            $estado = $boxController->cargarArchivo($idCarpeta, $archivo, $nombreSoporte);

            if ($estado['estado']) {
                $idArchivo = $estado['data']['entries'][0]['id'];
                PlanVisitasSuh3100::on($this->cliente)->where('plv_pk_id', $idVisita)->update(['plv_box_id_informe_final' => $idArchivo]);
            } else {
                throw new Exception('No fue posible cargar el informe final de la visita SUH 3100 '.$idVisita);
            }
        } else {
            $estado = $boxController->actualizarArchivo($idInforme, $archivo);

            if ($estado['estado']) {
                $idArchivo = $idInforme;
            } else {
                throw new Exception('No fue posible cargar el informe final de la visita SUH 3100 '.$idVisita);
            }
        }

        return $idArchivo;
    }
}
