<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

use App\Http\Services\DBClientConnection;

use App\Models\clientes\PlanVisitasPamec;
use App\Models\clientes\AuditoriaPamecAcreditacion;
use App\Models\clientes\AuditoriaPamecOtros;
use App\Models\clientes\ParametrosVerifica;
use App\Models\hospitalia\Clientes;

use App\Http\Controllers\API\BoxController;

use App\Jobs\IndicadoresPamecJob;

use Exception;

class PlanVisitasPamecController extends Controller
{
    private $cliente;

    function __construct() {
        // Crear la conexion temporal al esquema del cliente
        $this->middleware(function ($request, $next) {
            if (Auth::user() !== null) {
                $this->cliente = Auth::user()->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$this->cliente => $connection->getConnectionArray($this->cliente)]);
            }

            return $next($request);
        });
    }

    public function obtenerPlanVisitas(Request $request, $ano, $mes) {
        return PlanVisitasPamec::on($this->cliente)
                               ->selectRaw('plan_visitas_pamec.*, base.reps_prestadores.*, base.estados_visita.*, name')
                               ->join('base.users', 'usu_fk_id', 'id')
                               ->join('base.reps_prestadores', 'rpr_fk_id', 'rpr_pk_id')
                               ->leftJoin('base.estados_visita', 'esv_fk_id', 'esv_pk_id')
                               ->where('plv_ano', $ano)
                               ->where('plv_mes', $mes)
                               ->where('plv_eliminada', false)
                               ->orderBy('name', 'asc')
                               ->orderBy('plv_fecha_visita', 'asc')
                               ->orderBy('rpr_razon_social', 'asc')
                               ->get()->toArray();
    }

    public function obtenerPlanVisitasPorAuditor(Request $request, $auditor, $ano, $mes) {
        return PlanVisitasPamec::on($this->cliente)
                               ->join('base.reps_prestadores', 'rpr_fk_id', 'rpr_pk_id')
                               ->leftJoin('base.estados_visita', 'esv_fk_id', 'esv_pk_id')
                               ->where('plv_ano', $ano)
                               ->where('plv_mes', $mes)
                               ->where('usu_fk_id', $auditor)
                               ->where('plv_eliminada', false)
                               ->orderBy('plv_fecha_visita', 'asc')
                               ->orderBy('rpr_razon_social', 'asc')
                               ->get()->toArray();
    }

    public function obtenerPlanVisitasAnoPrestador(Request $request, $ano, $prestador) {
        return PlanVisitasPamec::on($this->cliente)
                               ->selectRaw('plv_pk_id, plv_ano, plv_mes, mun_fk_id, name, plv_fecha_visita, esv_descripcion, plv_fecha_cierre_visita')
                               ->join('base.reps_prestadores', 'rpr_fk_id', 'rpr_pk_id')
                               ->leftJoin('base.estados_visita', 'esv_fk_id', 'esv_pk_id')
                               ->leftJoin('base.users', 'usu_fk_id', 'id')
                               ->where('plv_ano', $ano)
                               ->where('rpr_fk_id', $prestador)
                               ->where('plv_eliminada', false)
                               ->where('plv_realizada', true)
                               ->where('plv_completada', true)
                               ->orderBy('plv_fecha_visita', 'asc')
                               ->orderBy('plv_pk_id')
                               ->get()->toArray();
    }

    public function buscarAsignacionPrestador(Request $request, $ano, $mes, $idPrestador) {
        return PlanVisitasPamec::on($this->cliente)
                               ->where('plv_ano', $ano)
                               ->where('plv_mes', $mes)
                               ->where('plv_eliminada', false)
                               ->where('rpr_fk_id', $idPrestador)
                               ->count();
    }

    public function obtenerSoporteVisita(Request $request, $idSoporte) {
        $boxController = new BoxController();
        $boxController->generarToken();

        $descarga = $boxController->descargarArchivo($idSoporte);

        if ($descarga['estado']) {
            return $descarga['file'];
        } else {
            throw new Exception('No se pudo descargar el archivo '.$idSoporte);
        }
    }

    public function crearPlanVisitas(Request $request) {
        $ano = $request->input('plv_ano');
        $mes = $request->input('plv_mes');

        DB::transaction(function() use($request, $ano, $mes) {
            $auditor = $request->input('auditor');
            $prestadoresAsignados = $request->input('prestadoresAsignados');

            foreach ($prestadoresAsignados as $rpr) {
                $id = PlanVisitasPamec::on($this->cliente)->create([
                    'plv_ano' => $ano,
                    'plv_mes' => $mes,
                    'usu_fk_id' => $auditor,
                    'rpr_fk_id' => $rpr['rpr_pk_id'],
                    'plv_comentario' => $rpr['plv_comentario'],
                    'plv_creada_por' => Auth::user()->id
                ])->plv_pk_id;

                // Crear la carpeta donde se guardarán los soportes de la visita
                $idCarpetaPamec = ParametrosVerifica::on($this->cliente)->first()->pav_box_id_pamec;
                $boxController = new BoxController();

                $token = $boxController->generarToken();

                if ($token['estado']) {
                    $estadoCarpeta = $boxController->crearFolder($id, $idCarpetaPamec);

                    if ($estadoCarpeta['estado']) {
                        PlanVisitasPamec::on($this->cliente)->where('plv_pk_id', $id)->update([
                            'plv_box_id_visita' => $estadoCarpeta['id']
                        ]);
                    } else {
                        throw new Exception('No fue posible crear la carpeta de la visita PAMEC con id '.$id);
                    }
                } else {
                    throw new Exception('No fue posible obtener un token para interactuar con BOX');
                }
            }
        });

        IndicadoresPamecJob::dispatch($ano, $mes)->onQueue('indicadores-pamec');
    }

    public function actualizarFechaVisita(Request $request, $idVisita) {
        PlanVisitasPamec::on($this->cliente)
                        ->where('plv_pk_id', $idVisita)
                        ->update(['plv_fecha_visita' => $request->input('plv_fecha_visita')]);
    }

    public function actualizarSincronizacionVisita(Request $request, $idVisita) {
        PlanVisitasPamec::on($this->cliente)->where('plv_pk_id', $idVisita)->update([
            'plv_visita_sincronizada' => $request->input('estado')
        ]);
    }

    public function finalizarVisita(Request $request) {
		$id = null;

        $infoVisita = json_decode($request->infoVisita, true);
        $instrumento = json_decode($request->instrumento, true);

        DB::connection($this->cliente)->transaction(function () use ($request, $id, $infoVisita, $instrumento) {
            // Finalizar visita
            PlanVisitasPamec::on($this->cliente)->where('plv_pk_id', $infoVisita['plv_fk_id'])->update([
                'plv_es_acreditacion' => isset($instrumento['codigo_instrumento']) ? ($instrumento['codigo_instrumento'] === 'PAMEC_ACREDITACION') : null,
                'plv_realizada' => true,
                'plv_completada' => isset($instrumento['aup_completado']) && intval($instrumento['aup_completado']) === 1 ? true : false,
                'plv_guardado_nube' => true,
                'plv_fecha_cierre_visita' => 'now()',
                'esv_fk_id' => $infoVisita['esv_fk_id'],
                'plv_comentario_cierre' => $infoVisita['plv_comentario_cierre'],
                'plv_evaluacion_1' => isset($instrumento['aup_instrumento']['aup_evaluacion_1']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_1']) === 1 ? true: false) : null,
                'plv_evaluacion_2' => isset($instrumento['aup_instrumento']['aup_evaluacion_2']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_2']) === 1 ? true: false) : null,
                'plv_evaluacion_3' => isset($instrumento['aup_instrumento']['aup_evaluacion_3']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_3']) === 1 ? true: false) : null,
                'plv_evaluacion_4' => isset($instrumento['aup_instrumento']['aup_evaluacion_4']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_4']) === 1 ? true: false) : null,
                'plv_evaluacion_5' => isset($instrumento['aup_instrumento']['aup_evaluacion_5']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_5']) === 1 ? true: false) : null,
                'plv_evaluacion_6' => isset($instrumento['aup_instrumento']['aup_evaluacion_6']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_6']) === 1 ? true: false) : null,
                'plv_evaluacion_7' => isset($instrumento['aup_instrumento']['aup_evaluacion_7']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_7']) === 1 ? true: false) : null,
                'plv_evaluacion_8' => isset($instrumento['aup_instrumento']['aup_evaluacion_8']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_8']) === 1 ? true: false) : null,
                'plv_evaluacion_9' => isset($instrumento['aup_instrumento']['aup_evaluacion_9']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_9']) === 1 ? true: false) : null,
                'plv_evaluacion_10' => isset($instrumento['aup_instrumento']['aup_evaluacion_10']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_10']) === 1 ? true: false) : null,
                'plv_resultado' => isset($instrumento['aup_instrumento']['aup_resultado']) ? $instrumento['aup_instrumento']['aup_resultado'] : null,
                'plv_total_preguntas' => isset($instrumento['aup_instrumento']['aup_total_preguntas']) ? $instrumento['aup_instrumento']['aup_total_preguntas'] : null,
                'plv_preguntas_cumplen' => isset($instrumento['aup_instrumento']['aup_total_preguntas']) ? $instrumento['aup_instrumento']['aup_preguntas_cumplen'] : null,
                'plv_preguntas_no_cumplen' => isset($instrumento['aup_instrumento']['aup_total_preguntas']) ? $instrumento['aup_instrumento']['aup_preguntas_no_cumplen'] : null
            ]);

            // Guardar instrumento
            if (isset($instrumento['codigo_instrumento']) && isset($instrumento['aup_instrumento'])) {
                $modeloPamec = null;

                if ($instrumento['codigo_instrumento'] === 'PAMEC_ACREDITACION') {
                    $modeloPamec = AuditoriaPamecAcreditacion::on($this->cliente);
                } else {
                    $modeloPamec = AuditoriaPamecOtros::on($this->cliente);
                }

                $posibleRegistroExistente = $modeloPamec->where('plv_fk_id', $infoVisita['plv_fk_id'])
                                                        ->where('usu_fk_id', $infoVisita['usu_fk_id'])
                                                        ->where('aup_token_local', $instrumento['aup_token_local'])
                                                        ->get()->toArray();

                $informacion = [
                    'iap_fk_id' => $instrumento['aup_instrumento']['iap_fk_id'],
                    'aup_instrumento' => json_encode($this->limpiarInstrumento($instrumento['aup_instrumento'])),
                    'aup_evaluacion_1' => isset($instrumento['aup_instrumento']['aup_evaluacion_1']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_1']) === 1 ? true: false) : null,
                    'aup_evaluacion_2' => isset($instrumento['aup_instrumento']['aup_evaluacion_2']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_2']) === 1 ? true: false) : null,
                    'aup_evaluacion_3' => isset($instrumento['aup_instrumento']['aup_evaluacion_3']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_3']) === 1 ? true: false) : null,
                    'aup_evaluacion_4' => isset($instrumento['aup_instrumento']['aup_evaluacion_4']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_4']) === 1 ? true: false) : null,
                    'aup_evaluacion_5' => isset($instrumento['aup_instrumento']['aup_evaluacion_5']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_5']) === 1 ? true: false) : null,
                    'aup_evaluacion_6' => isset($instrumento['aup_instrumento']['aup_evaluacion_6']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_6']) === 1 ? true: false) : null,
                    'aup_evaluacion_7' => isset($instrumento['aup_instrumento']['aup_evaluacion_7']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_7']) === 1 ? true: false) : null,
                    'aup_evaluacion_8' => isset($instrumento['aup_instrumento']['aup_evaluacion_8']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_8']) === 1 ? true: false) : null,
                    'aup_evaluacion_9' => isset($instrumento['aup_instrumento']['aup_evaluacion_9']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_9']) === 1 ? true: false) : null,
                    'aup_evaluacion_10' => isset($instrumento['aup_instrumento']['aup_evaluacion_10']) ? (intval($instrumento['aup_instrumento']['aup_evaluacion_10']) === 1 ? true: false) : null,
                    'aup_resultado' => isset($instrumento['aup_instrumento']['aup_resultado']) ? $instrumento['aup_instrumento']['aup_resultado'] : null,
                    'aup_observacion_visita' => $instrumento['aup_instrumento']['aup_observacion_visita'],
                    'aup_total_preguntas' => isset($instrumento['aup_instrumento']['aup_total_preguntas']) ? $instrumento['aup_instrumento']['aup_total_preguntas'] : null,
                    'aup_preguntas_cumplen' => isset($instrumento['aup_instrumento']['aup_total_preguntas']) ? $instrumento['aup_instrumento']['aup_preguntas_cumplen'] : null,
                    'aup_preguntas_no_cumplen' => isset($instrumento['aup_instrumento']['aup_total_preguntas']) ? $instrumento['aup_instrumento']['aup_preguntas_no_cumplen'] : null
                ];

                if ($instrumento['codigo_instrumento'] !== 'PAMEC_ACREDITACION') {
                    $informacion['aup_prg_enf_1'] = intval($instrumento['aup_instrumento']['aup_prg_enf_1']) === 1 ? true: false;
                    $informacion['aup_prg_enf_1_calificacion'] = $instrumento['aup_instrumento']['aup_prg_enf_1_calificacion'];
                    $informacion['aup_prg_enf_2'] = intval($instrumento['aup_instrumento']['aup_prg_enf_2']) === 1 ? true: false;
                    $informacion['aup_prg_enf_2_calificacion'] = $instrumento['aup_instrumento']['aup_prg_enf_2_calificacion'];
                    $informacion['aup_prg_enf_3'] = intval($instrumento['aup_instrumento']['aup_prg_enf_3']) === 1 ? true: false;
                    $informacion['aup_prg_enf_3_calificacion'] = $instrumento['aup_instrumento']['aup_prg_enf_3_calificacion'];
                }

                if (count($posibleRegistroExistente) > 0) {
                    $id = $posibleRegistroExistente[0]['aup_pk_id'];
                    $modeloPamec->where('aup_pk_id', $id)->update($informacion);
                } else {
                    $informacion['usu_fk_id'] = $infoVisita['usu_fk_id'];
                    $informacion['plv_fk_id'] = $infoVisita['plv_fk_id'];
                    $informacion['aup_ano'] = $infoVisita['plv_ano'];
                    $informacion['aup_mes'] = $infoVisita['plv_mes'];
                    $informacion['rpr_fk_id'] = $infoVisita['rpr_fk_id'];
                    $informacion['aup_token_local'] = $instrumento['aup_token_local'];

                    $id = $modeloPamec->create($informacion)->aup_pk_id;
                }

                // Carga de soportes en el Box
                $almacenamientoNube = Clientes::on('hospitalia')->where('cli_pk_id', $this->cliente)->pluck('cli_almacenamiento_nube')[0];

                if ($almacenamientoNube) {
                    $soportes = $request->file('soportes');

                    if ($soportes !== null) {
                        $idCarpeta = PlanVisitasPamec::on($this->cliente)->where('plv_pk_id', $infoVisita['plv_fk_id'])->pluck('plv_box_id_visita')[0];

                        $boxController = new BoxController();
                        $boxController->generarToken();

                        foreach ($soportes as $file) {
                            $nombreSoporte = $file->getClientOriginalName().'.pdf';
                            $estado = $boxController->cargarArchivo($idCarpeta, $file, $nombreSoporte);

                            if ($estado['estado']) {
                                PlanVisitasPamec::on($this->cliente)->where('plv_pk_id', $infoVisita['plv_fk_id'])->update([
                                    'plv_box_id_soporte' => $estado['data']['entries'][0]['id']
                                ]);
                            } else {
                                throw new Exception('No fue posible cargar el archivo correspondiente al servicio '.$nombreSoporte);
                            }
                        }
                    }
                }
            }
        });

        // Agregar la consolidación de los indicadores a la cola
        IndicadoresPamecJob::dispatch($infoVisita['plv_ano'], $infoVisita['plv_mes'])->onQueue('indicadores-pamec');

        return $id;
	}

    private function limpiarInstrumento($instrumento) {
        unset($instrumento['finalizado']);
        unset($instrumento['codigo_instrumento']);
        unset($instrumento['instrumento']);
        unset($instrumento['usu_fk_id']);
        unset($instrumento['tvi_fk_id']);
        unset($instrumento['plv_ano']);
        unset($instrumento['plv_mes']);
        unset($instrumento['aup_prg_enf_1']);
        unset($instrumento['aup_prg_enf_1_calificacion']);
        unset($instrumento['aup_prg_enf_2']);
        unset($instrumento['aup_prg_enf_2_calificacion']);
        unset($instrumento['aup_prg_enf_3']);
        unset($instrumento['aup_prg_enf_3_calificacion']);
        unset($instrumento['aup_token_local']);
        unset($instrumento['aup_evaluacion_1']);
        unset($instrumento['aup_evaluacion_2']);
        unset($instrumento['aup_evaluacion_3']);
        unset($instrumento['aup_evaluacion_4']);
        unset($instrumento['aup_evaluacion_5']);
        unset($instrumento['aup_evaluacion_6']);
        unset($instrumento['aup_evaluacion_7']);
        unset($instrumento['aup_evaluacion_8']);
        unset($instrumento['aup_evaluacion_9']);
        unset($instrumento['aup_evaluacion_10']);
        unset($instrumento['aup_resultado']);
        unset($instrumento['aup_observacion_visita']);
        unset($instrumento['aup_total_preguntas']);
        unset($instrumento['aup_preguntas_cumplen']);
        unset($instrumento['aup_preguntas_no_cumplen']);

        return $instrumento;
    }
}
