<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

use App\Http\Services\DBClientConnection;

use App\Models\clientes\InstrumentosAudCalidad;
use App\Models\clientes\HistoricoInsAudCalidad;

class InstrumentosAudCalidadController extends Controller
{
    private $cliente;

    function __construct() {
        // Crear la conexion temporal al esquema del cliente
        $this->middleware(function ($request, $next) {
            if (Auth::user() !== null) {
                $this->cliente = Auth::user()->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$this->cliente => $connection->getConnectionArray($this->cliente)]);
            }

            return $next($request);
        });
    }

    public function obtenerInstrumentosActivos(Request $request) {
        return InstrumentosAudCalidad::on($this->cliente)
                                     ->selectRaw('iac_pk_id, iac_version, iac_es_urgencias, iac_comentarios, iac_instrumento')
                                     ->where('iac_activa', true)
                                     ->orderBy('iac_es_urgencias')
                                     ->get()->toArray();
    }

    public function obtenerInstrumentosConUrgencias(Request $request) {
        return InstrumentosAudCalidad::on($this->cliente)
                                     ->where('iac_es_urgencias', true)
                                     ->orderBy('iac_activa', 'desc')
                                     ->orderBy('iac_pk_id', 'desc')
                                     ->get()->toArray();
    }

    public function obtenerInstrumentosSinUrgencias(Request $request) {
        return InstrumentosAudCalidad::on($this->cliente)
                                     ->where('iac_es_urgencias', false)
                                     ->orderBy('iac_activa', 'desc')
                                     ->orderBy('iac_pk_id', 'desc')
                                     ->get()->toArray();
    }

    public function crearInstrumento(Request $request) {
        $id = null;
        $nuevaVersion = null;

        DB::connection($this->cliente)->transaction(function() use($request, &$id, &$nuevaVersion) {
            $nuevaVersion = DB::connection($this->cliente)->select(
                $request->iac_es_urgencias ? 'select nextval(\'"'.$this->cliente.'".seq_aud_calidad_con_urgencias\')'
                                           : 'select nextval(\'"'.$this->cliente.'".seq_aud_calidad_sin_urgencias\')'
            )[0]->nextval;

            $usuario = Auth::user()->id;

            $id = InstrumentosAudCalidad::on($this->cliente)->create([
                'iac_version' => $nuevaVersion,
                'iac_es_urgencias' => $request->iac_es_urgencias,
                'iac_comentarios' => $request->iac_comentarios,
                'iac_instrumento' => json_encode($request->iac_instrumento),
                'iac_usu_creacion' => $usuario
            ])->iac_pk_id;

            HistoricoInsAudCalidad::on($this->cliente)->create([
                'iac_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hac_accion' => 'Creación de una nueva versión del instrumento'
            ]);
        });

        return [$id, $nuevaVersion];
    }

    public function actualizarInstrumento(Request $request, $id) {
        DB::connection($this->cliente)->transaction(function() use($request, &$id) {
            $usuario = Auth::user()->id;

            InstrumentosAudCalidad::on($this->cliente)->where('iac_pk_id', $id)->update([
                'iac_comentarios' => $request->iac_comentarios,
                'iac_instrumento' => $request->iac_instrumento
            ]);

            HistoricoInsAudCalidad::on($this->cliente)->create([
                'iac_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hac_accion' => 'Actualización del instrumento'
            ]);
        });
    }

    public function activarInstrumento(Request $request, $id) {
        DB::connection($this->cliente)->transaction(function() use($request, &$id) {
            $usuario = Auth::user()->id;
            $esUrgencias = $request->iac_es_urgencias;

            $idActivo = InstrumentosAudCalidad::on($this->cliente)
                                               ->where('iac_es_urgencias', $esUrgencias)
                                               ->where('iac_activa', true)
                                               ->pluck('iac_pk_id')[0];

            // Inactivar el actual
            InstrumentosAudCalidad::on($this->cliente)->where('iac_pk_id', $idActivo)->update([
                'iac_activa' => false
            ]);

            HistoricoInsAudCalidad::on($this->cliente)->create([
                'iac_fk_id' => $idActivo,
                'usu_fk_id' => $usuario,
                'hac_accion' => 'El instrumento pasa a estado inactivo'
            ]);

            // Activar el nuevo instrumento
            InstrumentosAudCalidad::on($this->cliente)->where('iac_pk_id', $id)->update([
                'iac_activa' => true
            ]);

            HistoricoInsAudCalidad::on($this->cliente)->create([
                'iac_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hac_accion' => 'El instrumento pasa a estado activo'
            ]);
        });
    }

    public function obtenerHistoricoVersion(Request $request, $id) {
        return HistoricoInsAudCalidad::on($this->cliente)
                                     ->selectRaw('historico_ins_aud_calidad.*, users.name')
                                     ->join('base.users', 'id', 'usu_fk_id')
                                     ->where('iac_fk_id', $id)
                                     ->orderBy('created_at', 'desc')
                                     ->get()->toArray();
    }
}
